<?php
namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Models\User;
use App\Models\UserReferral;
use App\Services\EmailTemplateService;
use App\Services\EmailVerificationService;
use App\Services\LoginHistoryService;
use App\Services\SettingsService;
use Elliptic\EC;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use kornrunner\Keccak;

class Web3WalletController extends Controller
{
    public function __construct(
        protected readonly LoginHistoryService $loginHistoryService,
    ){

    }
    const SUPPORTED_WALLETS = [
        'metamask' => 'MetaMask',
    ];


    /**
     * Validate referral code
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function validateReferral(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'referral_code' => 'required|string|max:255'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'valid' => false,
                'message' => 'Invalid referral code format'
            ], 400);
        }

        try {
            $referralCode = strtoupper(trim($request->referral_code));

            $referrer = User::where('referral_code', $referralCode)
                ->where('status', 'active')
                ->first();


            if (!$referrer) {
                return response()->json([
                    'success' => true,
                    'valid' => false,
                    'message' => 'Invalid referral code'
                ]);
            }

            $referralBonus = Setting::get('referral_bonus', null);
            return response()->json([
                'success' => true,
                'valid' => true,
                'message' => 'Valid referral code',
                'referral_info' => [
                    'referrer_id' => $referrer->id,
                    'referrer_name' => $referrer->name,
                    'referrer_email' => $referrer->email,
                    'bonus' => $referralBonus ? "You'll receive {$referralBonus} bonus!" : null
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error validating referral code', [
                'error' => $e->getMessage(),
                'referral_code' => $request->referral_code
            ]);

            return response()->json([
                'success' => false,
                'valid' => false,
                'message' => 'An error occurred while validating referral code'
            ], 500);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function checkSession(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'wallet_address' => 'required|string|regex:/^0x[a-fA-F0-9]{40}$/',
            'wallet_type' => 'required|string|in:' . implode(',', array_keys(self::SUPPORTED_WALLETS))
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'authenticated' => false
            ]);
        }

        $walletAddress = strtolower($request->wallet_address);
        if (Auth::check()) {
            $user = Auth::user();
            if (strtolower($user->wallet_address) === $walletAddress) {
                return response()->json([
                    'success' => true,
                    'authenticated' => true,
                    'user' => [
                        'id' => $user->id,
                        'name' => $user->name,
                        'wallet_address' => $user->wallet_address,
                    ]
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'authenticated' => false
        ]);
    }


    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function getNonce(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'wallet_address' => 'required|string|regex:/^0x[a-fA-F0-9]{40}$/',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid wallet address'
            ], 400);
        }

        $walletAddress = strtolower($request->wallet_address);
        $nonce = 'Sign this message to authenticate with your wallet: ' . Str::random(32);
        Cache::put("wallet_nonce:{$walletAddress}", $nonce, now()->addMinutes(5));

        return response()->json([
            'success' => true,
            'nonce' => $nonce,
            'message' => 'Please sign this message with your wallet'
        ]);
    }

    /**
     * @param string $walletAddress
     * @param string $message
     * @param string $signature
     * @return bool
     */
    private function verifySignature(string $walletAddress, string $message, string $signature): bool
    {
        try {
            $hash = Keccak::hash(sprintf("\x19Ethereum Signed Message:\n%s%s", strlen($message), $message), 256);
            $recoveredAddress = $this->recoverAddress($hash, $signature);
            return strtolower($recoveredAddress) === strtolower($walletAddress);

        } catch (\Exception $e) {
            Log::error('Signature verification failed', [
                'error' => $e->getMessage(),
                'wallet' => $walletAddress
            ]);
            return false;
        }
    }

    /**
     * @param string $hash
     * @param string $signature
     * @return string
     * @throws \Exception
     */
    private function recoverAddress(string $hash, string $signature): string
    {
        $signature = str_starts_with($signature, '0x') ? substr($signature, 2) : $signature;
        if (strlen($signature) !== 130) {
            throw new \Exception('Invalid signature length');
        }

        $r = substr($signature, 0, 64);
        $s = substr($signature, 64, 64);
        $v = hexdec(substr($signature, 128, 2));

        if ($v < 27) {
            $v += 27;
        }

        $ec = new EC('secp256k1');

        $recoveryId = $v - 27;
        $publicKey = $ec->recoverPubKey($hash, [
            'r' => $r,
            's' => $s
        ], $recoveryId);

        $publicKeyString = $publicKey->encode('hex');
        $publicKeyString = substr($publicKeyString, 2);

        $addressHash = Keccak::hash(hex2bin($publicKeyString), 256);
        return '0x' . substr($addressHash, -40);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function directLogin(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'wallet_address' => 'required|string|regex:/^0x[a-fA-F0-9]{40}$/',
            'signature' => 'required|string|min:132',
            'wallet_type' => 'required|string|in:' . implode(',', array_keys(self::SUPPORTED_WALLETS)),
            'referral_code' => 'nullable|string|max:255'
        ]);

        if ($validator->fails()) {
            $this->loginHistoryService->logAttempt($request->wallet_address ?? 'unknown', false, $request);

            return response()->json([
                'success' => false,
                'message' => 'Invalid input data'
            ], 400);
        }

        $walletAddress = strtolower($request->wallet_address);
        $signature = $request->signature;
        $walletType = $request->wallet_type;
        $referralCode = $request->referral_code;
        $nonce = Cache::get("wallet_nonce:{$walletAddress}");

        if (!$nonce) {
            $this->loginHistoryService->logAttempt($walletAddress, false, $request);

            return response()->json([
                'success' => false,
                'message' => 'Nonce expired or not found. Please request a new nonce.'
            ], 401);
        }

        if (!$this->verifySignature($walletAddress, $nonce, $signature)) {
            $this->loginHistoryService->logAttempt($walletAddress, false, $request);
            Cache::forget("wallet_nonce:{$walletAddress}");

            return response()->json([
                'success' => false,
                'message' => 'Invalid signature. Signature verification failed.'
            ], 401);
        }

        Cache::forget("wallet_nonce:{$walletAddress}");
        DB::beginTransaction();
        try {
            $existingUser = User::where('wallet_address', $walletAddress)->first();
            $isNewUser = !$existingUser;

            $referredBy = null;
            $referrer = null;

            if ($isNewUser && $referralCode) {
                $referrer = User::where('referral_code', $referralCode)->first();

                if ($referrer) {
                    $referredBy = $referrer->id;

                    Log::info('Referral code applied', [
                        'referral_code' => $referralCode,
                        'referrer_id' => $referrer->id,
                        'referrer_name' => $referrer->name,
                        'wallet_address' => $walletAddress
                    ]);
                }
            }

            $user = User::firstOrCreate(
                ['wallet_address' => $walletAddress],
                [
                    'uid' => 'UID' . strtoupper(Str::random(8)),
                    'name' => self::SUPPORTED_WALLETS[$walletType] . ' User',
                    'email' => 'wallet_' . substr($walletAddress, 2, 8) . '@noreply.local',
                    'password' => bcrypt(Str::random(32)),
                    'wallet_type' => $walletType,
                    'email_verified_at' => now(),
                    'role' => 'user',
                    'status' => 'active',
                    'referred_by' => $referredBy,
                    'referral_code' => strtoupper(Str::random(10)),
                ]
            );

            if ($isNewUser) {
                EmailTemplateService::sendTemplateEmail('welcome', $user, [
                    'user_name' => e($user->name),
                ]);

                if ($referrer) {
                    UserReferral::create([
                        'referrer_id' => $referrer->id,
                        'referred_id' => $user->id,
                        'level' => 1,
                        'status' => 'active',
                        'total_commission_earned' => 0,
                        'total_referral_investment' => 0,
                        'generation_count' => 0
                    ]);

                    $referrer->increment('total_referrals');
                    $referrer->increment('active_referrals');

                    Log::info('User referral relationship created', [
                        'referrer_id' => $referrer->id,
                        'referred_id' => $user->id,
                        'referral_code' => $referralCode
                    ]);
                }

                Log::info('New user created via Web3', [
                    'user_id' => $user->id,
                    'wallet_address' => $walletAddress,
                    'referred_by' => $referredBy,
                    'referrer_code' => $referralCode
                ]);
            }

            $user->update([
                'last_login_at' => now(),
                'wallet_type' => $walletType
            ]);

            if ($user->wallet()->exists()) {
                $user->wallet->update(['last_activity' => now()]);
            }

            DB::commit();
            Auth::login($user, true);
            $request->session()->regenerate();

            Log::info('User logged in via Web3', [
                'user_id' => $user->id,
                'wallet_address' => $walletAddress,
                'wallet_type' => $walletType,
                'is_new_user' => $isNewUser,
                'referred_by' => $referredBy,
                'user_agent' => $request->userAgent(),
                'ip' => $request->ip()
            ]);

            $this->loginHistoryService->logAttempt($user->email, true, $request);

            return response()->json([
                'success' => true,
                'message' => 'Login successful',
                'is_new_user' => $isNewUser,
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'wallet_address' => $user->wallet_address,
                    'wallet_type' => $user->wallet_type,
                    'referral_code' => $user->referral_code,
                ],
                'redirect_url' => route('user.dashboard')
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            $emailIdentifier = 'wallet_' . substr($walletAddress, 2, 8) . '@noreply.local';
            $this->loginHistoryService->logAttempt($emailIdentifier, false, $request);

            Log::error('Error in direct login', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'wallet_address' => $walletAddress,
                'line' => $e->getLine()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Login failed. Please try again.'
            ], 500);
        }
    }


    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function emailLogin(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|min:8',
        ]);

        if ($validator->fails()) {
            $this->loginHistoryService->logAttempt($request->email ?? 'unknown', false, $request);
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        if (!Auth::attempt($request->only('email', 'password'), true)) {
            $this->loginHistoryService->logAttempt($request->email, false, $request);
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        $user = Auth::user();
        $user->update(['last_login_at' => now()]);

        $request->session()->regenerate();
        Log::info('User logged in via email', [
            'user_id' => $user->id,
            'email' => $user->email,
            'ip' => $request->ip()
        ]);

        $this->loginHistoryService->logAttempt($user->email, true, $request);
        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'referral_code' => $user->referral_code,
            ],
            'redirect_url' => route('user.dashboard')
        ]);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function emailRegister(Request $request): JsonResponse
    {
        $request->merge(array_map(function($value) {
            return is_string($value) ? strip_tags(trim($value)) : $value;
        }, $request->all()));

        if (!SettingsService::isRegistrationEnabled()) {
            return response()->json([
                'success' => false,
                'message' => 'Registration is currently disabled.'
            ], 403);
        }

        $minPasswordLength = Setting::get('password_min_length', 8);

        $validator = Validator::make($request->all(), [
            'email' => 'required|email|unique:users,email',
            'password' => [
                'required',
                'string',
                "min:{$minPasswordLength}",
                'max:255',
                'confirmed'
            ],
            'referral_code' => 'nullable|string|max:255'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $email = strtolower(trim($request->email));
            $referredBy = null;
            $referrer = null;
            if ($request->referral_code) {
                $referrer = User::where('referral_code', $request->referral_code)->first();
                if ($referrer) {
                    $referredBy = $referrer->id;
                    Log::info('Referral code applied during registration', [
                        'referral_code' => $request->referral_code,
                        'referrer_id' => $referrer->id,
                        'email' => $email
                    ]);
                }
            }

            $user = User::create([
                'uid' => 'UID' . strtoupper(Str::random(8)),
                'name' => explode('@', $email)[0],
                'email' => $email,
                'password' => Hash::make($request->password),
                'role' => 'user',
                'status' => 'pending',
                'email_verified_at' => EmailVerificationService::isVerificationRequired() ? null : now(),
                'referred_by' => $referredBy,
                'referral_code' => strtoupper(Str::random(10)),
            ]);

            Log::info('User registered via email', [
                'user_id' => $user->id,
                'email' => $email,
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent()
            ]);

            if (EmailVerificationService::isVerificationRequired()) {
                EmailVerificationService::sendVerificationEmail($user);
                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => 'Registration successful! Please check your email and verify your account before logging in.',
                    'requires_verification' => true,
                    'redirect_url' => route('home')
                ]);
            } else {
                $user->update([
                    'status' => 'active',
                    'email_verified_at' => now()
                ]);
            }

            if ($referrer) {
                UserReferral::create([
                    'referrer_id' => $referrer->id,
                    'referred_id' => $user->id,
                    'level' => 1,
                    'status' => 'active',
                    'total_commission_earned' => 0,
                    'total_referral_investment' => 0,
                    'generation_count' => 0
                ]);

                $referrer->increment('total_referrals');
                $referrer->increment('active_referrals');

                Log::info('Referral relationship created', [
                    'referrer_id' => $referrer->id,
                    'referred_id' => $user->id,
                    'referral_code' => $request->referral_code
                ]);
            }

            if (!EmailVerificationService::isVerificationRequired()) {
                EmailTemplateService::sendTemplateEmail('welcome', $user, [
                    'user_name' => e($user->name),
                ]);
            }

            $user->update(['last_login_at' => now()]);
            DB::commit();

            Auth::login($user, true);
            $request->session()->regenerate();

            $this->loginHistoryService->logAttempt($user->email, true, $request);

            return response()->json([
                'success' => true,
                'message' => 'Registration successful',
                'requires_verification' => false,
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'referral_code' => $user->referral_code,
                ],
                'redirect_url' => route('user.dashboard')
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Error in email registration', [
                'email' => $request->email,
                'error' => $e->getMessage(),
                'line' => $e->getLine(),
                'ip' => $request->ip()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred during registration. Please try again.'
            ], 500);
        }
    }

    /**
     * @param $signature
     * @return bool
     */
    private function isValidSignature($signature): bool
    {
        return (
            is_string($signature) &&
            strlen($signature) >= 130 &&
            str_starts_with($signature, '0x') &&
            ctype_xdigit(substr($signature, 2))
        );
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function logout(Request $request): RedirectResponse
    {
        try {
            $userId = Auth::id();
            $walletAddress = Auth::user()?->wallet_address;

            Auth::logout();

            $request->session()->invalidate();
            $request->session()->regenerateToken();

            Log::info('User logged out', [
                'user_id' => $userId,
                'wallet_address' => $walletAddress
            ]);

            return redirect()->route('home')->with('success', 'Logged out successfully');

        } catch (\Exception $e) {
            Log::error('Error during logout', [
                'error' => $e->getMessage()
            ]);

            return back()->with('error', 'Logout failed');
        }
    }
}
