<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Language;
use App\Models\Translation;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Inertia\Inertia;
use Inertia\Response;

class LanguageController extends Controller
{
    /**
     * @return Response
     */
    public function index(): Response
    {
        try {
            $languages = Language::withCount('translations')
                ->orderBy('is_default', 'desc')
                ->orderBy('name')
                ->get();

            return Inertia::render('Admin/Language/Index', [
                'languages' => $languages,
            ]);

        } catch (\Exception $e) {
            return Inertia::render('Admin/Language/Index', [
                'languages' => [],
                'error' => 'Unable to load languages. Please try again.'
            ]);
        }
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'code' => 'required|string|max:10|unique:languages,code',
            'name' => 'required|string|max:255',
            'native_name' => 'required|string|max:255',
            'flag' => 'required|string|max:10',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
        ]);

        try {
            Cache::forget('frontend_languages');
            $language = Language::create($request->all());
            $this->createBaseTranslations($language);
            return redirect()->back()->with('success', 'Language created successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Failed to create language. Please try again.']);
        }
    }

    /**
     * @param Request $request
     * @param Language $language
     * @return RedirectResponse
     */
    public function update(Request $request, Language $language): RedirectResponse
    {
        $request->validate([
            'code' => 'required|string|max:10|unique:languages,code,' . $language->id,
            'name' => 'required|string|max:255',
            'native_name' => 'required|string|max:255',
            'flag' => 'required|string|max:10',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
        ]);

        try {
            Cache::forget('frontend_languages');
            $language->update($request->all());
            return redirect()->back()->with('success', 'Language updated successfully.');

        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Failed to update language. Please try again.']);
        }
    }

    /**
     * @param Language $language
     * @return RedirectResponse
     */
    public function destroy(Language $language): RedirectResponse
    {
        try {
            if ($language->is_default) {
                return redirect()->back()->withErrors(['error' => 'Cannot delete default language.']);
            }

            $language->translations()->delete();
            $language->delete();

            return redirect()->back()->with('success', 'Language deleted successfully.');

        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Failed to delete language. Please try again.']);
        }
    }

    public function translations(Request $request, Language $language): Response
    {
        try {
            $search = $request->get('search', '');
            $status = $request->get('status', 'all');
            $component = $request->get('component', 'header');
            $perPage = $request->get('per_page', 20);

            $query = Translation::where('language_id', $language->id);
            if ($component === 'dynamic') {
                $query->where('is_dynamic', 1);
            } else {
                $componentKeys = $this->getComponentKeys($component);
                if (!empty($componentKeys)) {
                    $query->whereIn('key', $componentKeys);
                } else {
                    $query->where('id', 0);
                }
            }

            if ($search && trim($search) !== '') {
                $searchTerm = trim($search);
                $query->where(function($q) use ($searchTerm) {
                    $q->where('key', 'like', "%{$searchTerm}%")
                        ->orWhere('value', 'like', "%{$searchTerm}%");
                });
            }

            if ($status !== 'all') {
                if ($status === 'translated') {
                    $query->whereNotNull('value')
                        ->where('value', '!=', '');
                } elseif ($status === 'untranslated') {
                    $query->where(function($q) {
                        $q->whereNull('value')
                            ->orWhere('value', '=', '');
                    });
                }
            }

            $translations = $query->orderBy('key')->paginate($perPage);
            return Inertia::render('Admin/Language/Translations', [
                'language' => $language,
                'translations' => $translations,
                'filters' => [
                    'search' => $search,
                    'status' => $status,
                    'component' => $component,
                    'per_page' => $perPage,
                ],
                'components' => $this->getAvailableComponents(),
            ]);

        } catch (\Exception $e) {
            return Inertia::render('Admin/Language/Translations', [
                'language' => $language,
                'translations' => [],
                'filters' => [
                    'search' => '',
                    'status' => 'all',
                    'component' => 'header',
                    'per_page' => 20,
                ],
                'components' => $this->getAvailableComponents(),
                'error' => 'Unable to load translations. Please try again.'
            ]);
        }
    }

    public function updateTranslation(Request $request, Language $language, Translation $translation): RedirectResponse
    {
        $request->validate([
            'value' => 'required|string',
        ]);

        try {
            $translation->update([
                'value' => $request->value,
            ]);

            Cache::forget("translations.{$language->code}");

            return redirect()->back()->with('success', 'Translation updated successfully.');

        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Failed to update translation. Please try again.']);
        }
    }

    private function createBaseTranslations(Language $language): void
    {
        $baseKeys = array_merge(
            $this->getComponentKeys('general'),
            $this->getComponentKeys('dashboard'),
            $this->getComponentKeys('transactions'),
            $this->getComponentKeys('earnings'),
            $this->getComponentKeys('deposit'),
            $this->getComponentKeys('investment'),
            $this->getComponentKeys('mining'),
            $this->getComponentKeys('staking'),
            $this->getComponentKeys('withdrawal'),
            $this->getComponentKeys('profile'),
            $this->getComponentKeys('referrals'),
            $this->getComponentKeys('network'),
            $this->getComponentKeys('header'),
            $this->getComponentKeys('footer'),
            $this->getComponentKeys('support'),
            $this->getComponentKeys('kyc_verification'),
            $this->getComponentKeys('login_history'),
            $this->getComponentKeys('password_change'),
            $this->getComponentKeys('two_factor_auth'),
            $this->getComponentKeys('ticket_details'),
            $this->getComponentKeys('subscription'),
            $this->getComponentKeys('miscellaneous')
        );

        $uniqueKeys = array_unique($baseKeys);

        foreach ($uniqueKeys as $key) {
            Translation::create([
                'language_id' => $language->id,
                'key' => $key,
                'value' => '',
                'is_dynamic' => 0,
            ]);
        }
    }

    private function getAvailableComponents(): array
    {
        return [
            ['value' => 'general', 'label' => 'General & Common'],
            ['value' => 'dashboard', 'label' => 'Dashboard'],
            ['value' => 'transactions', 'label' => 'Transactions'],
            ['value' => 'earnings', 'label' => 'Earnings'],
            ['value' => 'deposit', 'label' => 'Deposit'],
            ['value' => 'investment', 'label' => 'Investment'],
            ['value' => 'mining', 'label' => 'Mining'],
            ['value' => 'staking', 'label' => 'Staking'],
            ['value' => 'withdrawal', 'label' => 'Withdrawal'],
            ['value' => 'profile', 'label' => 'Profile'],
            ['value' => 'referrals', 'label' => 'Referrals'],
            ['value' => 'network', 'label' => 'Network'],
            ['value' => 'header', 'label' => 'Header'],
            ['value' => 'footer', 'label' => 'Footer'],
            ['value' => 'support', 'label' => 'Support'],
            ['value' => 'kyc_verification', 'label' => 'KYC Verification'],
            ['value' => 'login_history', 'label' => 'Login History'],
            ['value' => 'password_change', 'label' => 'Password Change'],
            ['value' => 'two_factor_auth', 'label' => 'Two Factor Authentication'],
            ['value' => 'ticket_details', 'label' => 'Ticket Details'],
            ['value' => 'subscription', 'label' => 'Subscription'],
            ['value' => 'miscellaneous', 'label' => 'Miscellaneous'],
            ['value' => 'dynamic', 'label' => 'Dynamic Content'],
        ];
    }

    private function getComponentKeys(string $component): array
    {
        $components = [
            'general' => [
                'trustedBy', 'organizations', 'search', 'filters', 'clear', 'refresh', 'details', 'action',
                'previous', 'next', 'page', 'currentPage', 'goToPage', 'pagination', 'goToPreviousPage',
                'goToNextPage', 'currentPageNumber', 'goToPageNumber', 'viewAll', 'goToDashboard',
                'closeModal', 'total', 'goal', 'level', 'copy', 'copied', 'of', 'to', 'showing', 'results',
                'showingResults', 'view', 'viewDetails', 'close', 'cancel', 'cancelling', 'create', 'id',
                'all', 'pending', 'cancelled', 'completed', 'processing', 'failed', 'success', 'created',
                'approved', 'rejected', 'notAvailable', 'invalidDate', 'invalidTime', 'unknown',
                'copiedToClipboard', 'show', 'data', 'filterBy', 'records', 'recordsFound', 'from', 'with',
                'about', 'on', 'for'
            ],

            'dashboard' => [
                'dashboard', 'overview', 'portfolioValue', 'investmentDistribution', 'assetAllocation',
                'distributionPieChart', 'doughnutChart', 'noDistributionData', 'distributionLegend',
                'colorIndicator', 'investmentsActions', 'activeInvestmentPlans', 'monitorInvestments',
                'activeInvestmentsList', 'investmentPlan', 'investmentAmount', 'dashboardRefreshed',
                'failedToRefresh', 'investmentDashboard', 'portfolioStatistics', 'portfolioBalance',
                'dailyChange', 'plus', 'minus', 'percent', 'last24h', 'totalInvested', 'activePlans',
                'roi', 'referralIncome', 'portfolioOverview', 'realTimeAssetManagement',
                'totalPortfolioValue', 'dailyYield24h', 'growthTrend', 'portfolioActive', 'updated',
                'secondsAgo', 'chartsAnalytics', 'portfolioPerformance', 'realTimeTracking',
                'chartTimeframe', 'portfolioChart', 'lineChart', 'dailyRate', 'daily', 'complete',
                'noActiveInvestments', 'quickActions', 'quickActionLinks', 'navigateDeposit',
                'navigateInvest', 'navigateWithdraw', 'aiInsights', 'aiRecommendationsList',
                'aiRecommendation', 'confidence', 'confidenceScore', 'learnMore', 'transactionsReferrals',
                'transactionsList'
            ],

            'transactions' => [
                'withdrawal', 'credit', 'debit', 'transfer', 'payment', 'refund', 'noTransactionsYet',
                'transactions', 'transactionStatistics', 'completedTransactions',
                'transactionIdPlaceholder', 'allTypes', 'startDate', 'endDate', 'applyFilters',
                'refreshingTransactions', 'refreshTransactions', 'transactionHistory',
                'totalTransactionsCount', 'transactionHistoryTable', 'walletType', 'postBalance',
                'viewDetailsFor', 'noTransactionsFound', 'transactionDetails',
                'transactionHistoryRefreshed', 'failedToRefreshHistory', 'transactionId', 'trxId',
                'transaction', 'transactionInformation', 'timeline', 'recentTransactions',
                'latestActivity', 'totalTransactions', 'amountDetails'
            ],

            'earnings' => [
                'earnings', 'financials', 'earningsDashboard', 'earningsStatistics', 'investmentProfit',
                'stakes', 'earningsSummary', 'thisMonth', 'totalEarnings', 'earningsBreakdown',
                'earningsHistory', 'last30Days', 'investmentEarnings', 'stakingRewards',
                'referralBonuses', 'miningRewards', 'interest', 'bonus', 'referral', 'mining', 'staking',
                'investment', 'other', 'amount', 'date', 'type', 'status', 'description', 'noEarningsYet',
                'startInvestingToEarn', 'earningsChart', 'monthlyEarnings', 'loadingEarnings',
                'failedToLoadEarnings', 'earningsType', 'earningsDetails'
            ],

            'deposit' => [
                'deposit', 'deposits', 'depositFunds', 'quickDepositOptions', 'depositAmount',
                'enterDepositAmount', 'selectPaymentMethod', 'depositHistory', 'depositStatistics',
                'totalDeposits', 'pendingDeposits', 'minimumDeposit', 'maximumDeposit',
                'estimatedArrival', 'instantDeposit', 'processingTime', 'hours', 'minutes', 'instant',
                'makeDeposit', 'depositInstructions', 'scanQrCode', 'copyAddress', 'sendExactAmount',
                'confirmTransaction', 'walletAddress', 'scanQrCodeToDeposit', 'clickToCopyAddress',
                'addressCopied', 'failedToCopyAddress', 'network', 'minimumAmount', 'fees',
                'estimatedTime', 'depositNow', 'processing', 'confirming', 'depositSuccess',
                'depositPending', 'depositFailed', 'noDepositsYet', 'makeFirstDeposit',
                'recentDeposits', 'depositAmountRequired', 'depositAmountTooLow', 'depositAmountTooHigh',
                'depositSubmitted', 'depositSubmissionFailed', 'selectNetwork', 'selectMethod',
                'transactionHash', 'uploadProof', 'optional', 'depositDetails', 'paymentMethod',
                'bankTransfer', 'creditCard', 'cryptocurrency', 'depositNote', 'depositProcessingNote',
                'cryptocurrencyDeposits', 'depositLimits'
            ],

            'investment' => [
                'investment', 'investments', 'investNow', 'invest', 'myInvestments',
                'investmentPlans', 'investmentOpportunities', 'availablePlans', 'planDetails',
                'selectPlan', 'minimumInvestment', 'maximumInvestment', 'duration', 'expectedReturn',
                'dailyReturn', 'monthlyReturn', 'annualReturn', 'totalReturn', 'investmentPeriod',
                'days', 'weeks', 'months', 'years', 'flexible', 'fixed', 'planType', 'riskLevel',
                'low', 'medium', 'high', 'riskFree', 'moderate', 'aggressive', 'features',
                'investmentFeatures', 'minimumDeposit', 'maximumDeposit', 'earlyWithdrawal',
                'autoReinvest', 'compoundInterest', 'principalReturn', 'capitalProtection',
                'liquidityOption', 'investmentStatistics', 'totalInvested', 'activeInvestments',
                'completedInvestments', 'totalReturns', 'averageROI', 'investmentHistory',
                'recentInvestments', 'investmentDetails', 'startDate', 'endDate', 'investedAmount',
                'currentValue', 'profitLoss', 'profitEarned', 'remainingDays', 'daysCompleted',
                'progressPercentage', 'investmentStatus', 'active', 'matured', 'cancelled',
                'investmentActions', 'viewDetails', 'reinvest', 'withdraw', 'extendPlan',
                'cancelInvestment', 'confirmInvestment', 'investmentConfirmation',
                'reviewInvestmentDetails', 'investmentAmount', 'selectAmount', 'enterAmount',
                'amountToInvest', 'estimatedProfit', 'maturityDate', 'maturityValue',
                'confirmInvest', 'makePayment', 'paymentDetails', 'agreeToTerms', 'termsAndConditions',
                'investmentSuccess', 'investmentCreated', 'investmentPending', 'investmentFailed',
                'investmentCancelled', 'noInvestmentsYet', 'startInvesting', 'exploreOpportunities',
                'investmentAmountRequired', 'investmentAmountTooLow', 'investmentAmountTooHigh',
                'insufficientBalance', 'investmentSubmitted', 'investmentSubmissionFailed',
                'selectInvestmentPlan', 'planName', 'returnRate', 'compounding', 'compoundingFrequency',
                'interestType', 'simpleInterest', 'compound', 'payoutFrequency', 'payout',
                'maturity', 'endOfTerm', 'totalInvestmentValue', 'projectedEarnings',
                'investmentCalculator', 'calculateReturns', 'investmentBreakdown', 'principal',
                'interestEarned', 'totalAmount', 'whatYouGet', 'howItWorks', 'investmentProcess',
                'step1', 'step2', 'step3', 'step4', 'chooseYourPlan', 'selectInvestmentAmount',
                'confirmYourInvestment', 'earnReturns'
            ],

            'mining' => [
                'mining', 'startMining', 'stopMining', 'miningDashboard', 'miningStatistics',
                'hashRate', 'miningPower', 'totalMined', 'todaysMining', 'activeMiner',
                'miningStatus', 'miningActive', 'miningInactive', 'miningPaused', 'miningCompleted',
                'miningProgress', 'timeRemaining', 'miningSpeed', 'miningEfficiency',
                'hashesPerSecond', 'kilohashesPerSecond', 'megahashesPerSecond', 'gigahashesPerSecond',
                'miningRewards', 'dailyRewards', 'weeklyRewards', 'monthlyRewards', 'totalRewards',
                'estimatedEarnings', 'currentHashRate', 'networkHashRate', 'difficulty',
                'blockReward', 'miningPool', 'poolShare', 'miningHistory', 'recentMining',
                'miningEvents', 'blocksMined', 'sharesSubmitted', 'miningUptime', 'uptimePercentage',
                'miningEquipment', 'hardwareStatus', 'temperature', 'powerConsumption',
                'coolingStatus', 'miningSettings', 'autoStart', 'miningIntensity',
                'powerMode', 'lowPower', 'balanced', 'highPerformance', 'coolingMode',
                'miningNotifications', 'notifyOnComplete', 'notifyOnErrors', 'notifyOnMilestones',
                'miningTiers', 'freeTier', 'basicTier', 'premiumTier', 'unlockTier',
                'upgradeMiner', 'purchaseMiner', 'minerDetails', 'minerType', 'minerCapacity',
                'minerLevel', 'minerBoost', 'boostMultiplier', 'activateBoost', 'boostActive',
                'boostRemaining', 'noActiveMiner', 'selectMinerToStart', 'startYourMining',
                'miningSuccess', 'miningStarted', 'miningStopped', 'miningPausedSuccess',
                'miningResumed', 'miningError', 'miningFailed', 'insufficientMiningPower',
                'minerCooldown', 'minerNotAvailable', 'calculateMiningProfit', 'miningCalculator',
                'miningProfitability', 'electricityCost', 'hardwareCost', 'maintenanceCost',
                'netProfit', 'breakEvenTime', 'roiPeriod', 'miningFAQ', 'whatIsMining',
                'howToStartMining', 'miningRequirements', 'miningRisks', 'miningSupport',
                'contactSupport', 'startMiningJourney'
            ],

            'staking' => [
                'staking', 'stake', 'unstake', 'stakingDashboard', 'stakingStatistics',
                'totalStaked', 'stakingRewards', 'stakingBalance', 'availableToStake',
                'stakedAmount', 'stakingAPY', 'stakingAPR', 'annualPercentageYield',
                'annualPercentageRate', 'stakingPools', 'stakingOptions', 'selectPool',
                'poolDetails', 'poolName', 'poolSize', 'participants', 'minimumStake',
                'maximumStake', 'lockPeriod', 'lockupPeriod', 'unlockDate', 'flexibleStaking',
                'lockedStaking', 'stakingDuration', 'stakingType', 'stakingTier', 'tierBenefits',
                'rewardRate', 'rewardFrequency', 'dailyRewards', 'weeklyRewards', 'monthlyRewards',
                'claimRewards', 'rewardsClaimed', 'pendingRewards', 'accumulatedRewards',
                'compoundRewards', 'autoCompound', 'rewardsHistory', 'stakingHistory',
                'recentStaking', 'stakingEvents', 'stakingStatus', 'activeStaking', 'stakingPending',
                'stakingCompleted', 'stakingCancelled', 'stakingActions', 'stakeNow', 'addStake',
                'increaseStake', 'unstakeNow', 'withdrawStake', 'emergencyUnstake',
                'extendStaking', 'renewStaking', 'stakingCalculator', 'calculateStakingRewards',
                'estimatedRewards', 'stakingProjection', 'amountToStake', 'selectStakingAmount',
                'enterStakingAmount', 'stakingAmountRequired', 'stakingAmountTooLow',
                'stakingAmountTooHigh', 'insufficientBalance', 'confirmStaking',
                'stakingConfirmation', 'reviewStakingDetails', 'stakingTerms', 'earlyUnstakePenalty',
                'penaltyPercentage', 'penaltyAmount', 'agreeToStakingTerms', 'stakingSuccess',
                'stakingInitiated', 'stakingFailed', 'unstakingSuccess', 'unstakingInitiated',
                'unstakingFailed', 'rewardsClaimSuccess', 'rewardsClaimFailed', 'noActiveStaking',
                'startStakingToday', 'exploreStakingPools', 'stakingBenefits', 'passiveIncome',
                'earnWhileYouSleep', 'compoundGrowth', 'networkSecurity', 'governanceRights',
                'votingPower', 'stakingRisks', 'lockupRisk', 'marketVolatility', 'slashingRisk',
                'liquidityRisk', 'stakingFAQ', 'whatIsStaking', 'howStakingWorks',
                'stakingRequirements', 'stakingRewardsInfo', 'unstakingProcess',
                'stakingSupport', 'stakingGuide', 'beginnerFriendly', 'stakingStrategy',
                'diversifyStaking', 'stakingPerformance', 'performanceMetrics', 'stakingYield',
                'effectiveYield', 'stakingBonus', 'loyaltyBonus', 'earlyBirdBonus',
                'stakingLeaderboard', 'topStakers', 'stakingRank', 'stakingLevel',
                'levelUpRewards', 'stakingMilestones', 'achievementUnlocked', 'stakingBadges',
                'stakingNotifications', 'notifyRewards', 'notifyUnlock', 'notifyMilestones'
            ],

            'withdrawal' => [
                'withdrawal', 'withdrawals', 'withdraw', 'withdrawFunds', 'withdrawalRequest',
                'requestWithdrawal', 'withdrawalDashboard', 'withdrawalStatistics',
                'totalWithdrawals', 'pendingWithdrawals', 'completedWithdrawals',
                'failedWithdrawals', 'withdrawalHistory', 'recentWithdrawals', 'withdrawalDetails',
                'withdrawalAmount', 'enterWithdrawalAmount', 'selectWithdrawalMethod',
                'withdrawalMethod', 'withdrawalDestination', 'destinationAddress',
                'bankAccount', 'cryptoWallet', 'minimumWithdrawal', 'maximumWithdrawal',
                'dailyWithdrawalLimit', 'weeklyWithdrawalLimit', 'monthlyWithdrawalLimit',
                'availableBalance', 'withdrawableBalance', 'lockedBalance', 'withdrawalFees',
                'processingFee', 'networkFee', 'totalFees', 'amountAfterFees', 'netAmount',
                'youWillReceive', 'estimatedArrival', 'processingTime', 'instantWithdrawal',
                'standardWithdrawal', 'withdrawalSpeed', 'fast', 'standard', 'slow',
                'withdrawalInstructions', 'provideWithdrawalDetails', 'enterDestinationAddress',
                'selectNetwork', 'selectChain', 'confirmWithdrawalAmount', 'reviewWithdrawal',
                'verifyDetails', 'confirmWithdrawal', 'withdrawalConfirmation', 'withdrawalOTP',
                'enter2FACode', 'enterVerificationCode', 'securityVerification',
                'withdrawalTerms', 'agreeToWithdrawalTerms', 'withdrawalNote',
                'withdrawalWarning', 'irreversibleTransaction', 'doubleCheckDetails',
                'withdrawalSuccess', 'withdrawalSubmitted', 'withdrawalPending',
                'withdrawalProcessing', 'withdrawalCompleted', 'withdrawalFailed',
                'withdrawalCancelled', 'withdrawalRejected', 'cancelWithdrawal',
                'cancelPendingWithdrawal', 'retryWithdrawal', 'withdrawalReason',
                'reasonForRejection', 'noWithdrawalsYet', 'makeFirstWithdrawal',
                'withdrawalAmountRequired', 'withdrawalAmountTooLow', 'withdrawalAmountTooHigh',
                'insufficientBalance', 'withdrawalLimitExceeded', 'addressRequired',
                'invalidAddress', 'addressVerificationFailed', 'withdrawalSubmissionFailed',
                'withdrawalCannotBeCancelled', 'withdrawalCancellationSuccess',
                'withdrawalCancellationFailed', 'withdrawalFAQ', 'withdrawalProcessInfo',
                'withdrawalFeesInfo', 'withdrawalLimitsInfo', 'withdrawalTimeframes',
                'withdrawalSecurity', 'withdrawalSupport', 'withdrawalGuide',
                'withdrawalRequirements', 'kycRequired', 'verifyAccount', 'accountNotVerified',
                'completeKYC', 'twoFactorRequired', 'enable2FA', 'withdrawalRestrictions',
                'accountLocked', 'contactSupport', 'withdrawalMethods', 'supportedMethods',
                'methodNotAvailable', 'selectDifferentMethod'
            ],

            'profile' => [
                'profile', 'myProfile', 'profileSettings', 'accountSettings', 'personalInfo',
                'personalInformation', 'fullName', 'firstName', 'lastName', 'username',
                'emailAddress', 'email', 'phoneNumber', 'phone', 'dateOfBirth', 'gender',
                'male', 'female', 'other', 'address', 'city', 'state', 'country', 'postalCode',
                'zipCode', 'nationality', 'profilePicture', 'avatar', 'uploadPhoto',
                'changePhoto', 'removePhoto', 'updateProfile', 'saveChanges', 'editProfile',
                'accountInformation', 'accountDetails', 'userId', 'accountId', 'memberSince',
                'accountStatus', 'accountType', 'verificationStatus', 'verified', 'unverified',
                'partiallyVerified', 'verifyNow', 'completeVerification', 'accountBalance',
                'walletBalance', 'totalBalance', 'availableBalance', 'portfolioValue',
                'accountStatistics', 'securitySettings', 'security', 'changePassword',
                'currentPassword', 'newPassword', 'confirmPassword', 'passwordUpdated',
                'twoFactorAuthentication', 'enable2FA', 'disable2FA', 'twoFAEnabled',
                'twoFADisabled', 'securityQuestions', 'updateSecurityQuestions', 'loginHistory',
                'recentLogins', 'viewLoginHistory', 'notificationSettings', 'notifications',
                'emailNotifications', 'smsNotifications', 'pushNotifications',
                'notificationPreferences', 'accountNotifications', 'transactionAlerts',
                'withdrawalAlerts', 'depositAlerts', 'securityAlerts', 'marketingEmails',
                'newsletterSubscription', 'privacySettings', 'privacy', 'publicProfile',
                'privateProfile', 'showEmail', 'hideEmail', 'showPhone', 'hidePhone',
                'profileVisibility', 'languagePreferences', 'language', 'selectLanguage',
                'timeZone', 'selectTimeZone', 'currency', 'preferredCurrency', 'theme',
                'darkMode', 'lightMode', 'autoTheme', 'displaySettings', 'deleteAccount',
                'deactivateAccount', 'closeAccount', 'accountDeletion', 'deleteAccountWarning',
                'confirmAccountDeletion', 'enterPasswordToDelete', 'accountDeleted',
                'accountDeactivated', 'profileUpdated', 'profileUpdateFailed', 'uploadFailed',
                'invalidFileType', 'fileTooLarge', 'maxFileSize', 'requiredField',
                'invalidEmail', 'invalidPhone', 'invalidDate', 'updateFailed',
                'savingChanges', 'loading', 'cancel', 'save', 'edit', 'delete', 'confirm',
                'back', 'next', 'finish', 'skip'
            ],

            'referrals' => [
                'referrals', 'referralProgram', 'referFriends', 'referralDashboard',
                'referralStatistics', 'totalReferrals', 'activeReferrals', 'referralEarnings',
                'totalEarned', 'pendingEarnings', 'referralBonus', 'referralCommission',
                'referralCode', 'yourReferralCode', 'shareReferralCode', 'copyReferralCode',
                'referralLink', 'yourReferralLink', 'shareReferralLink', 'copyReferralLink',
                'codeCopied', 'linkCopied', 'shareVia', 'shareViaEmail', 'shareViaSocial',
                'shareOnFacebook', 'shareOnTwitter', 'shareOnWhatsApp', 'shareOnTelegram',
                'referralHistory', 'referralList', 'yourReferrals', 'referredUsers',
                'referralDetails', 'referredBy', 'referralDate', 'referralStatus',
                'referralLevel', 'level1', 'level2', 'level3', 'level4', 'level5',
                'directReferrals', 'indirectReferrals', 'teamReferrals', 'downlineStructure',
                'referralTree', 'genealogyTree', 'networkTree', 'viewTree', 'expandTree',
                'collapseTree', 'referralCommissionRates', 'commissionStructure',
                'level1Commission', 'level2Commission', 'level3Commission', 'level4Commission',
                'level5Commission', 'lifetimeCommission', 'referralRequirements',
                'minimumReferrals', 'qualifiedReferrals', 'activeReferralsRequired',
                'referralRewards', 'referralIncentives', 'bonusTiers', 'achievementBonuses',
                'milestoneBonuses', 'referralContest', 'leaderboard', 'topReferrers',
                'yourRank', 'rankingsUpdated', 'referralPerformance', 'conversionRate',
                'clickThroughRate', 'signupRate', 'activationRate', 'referralAnalytics',
                'referralMetrics', 'totalClicks', 'totalSignups', 'totalActivations',
                'totalInvestments', 'referralValue', 'lifetimeValue', 'averageValue',
                'referralFAQ', 'howItWorks', 'referralTerms', 'referralConditions',
                'eligibilityCriteria', 'payoutSchedule', 'payoutMethod', 'minimumPayout',
                'noReferralsYet', 'startReferring', 'inviteFriends', 'growYourNetwork',
                'referralSuccess', 'referralRegistered', 'referralActive', 'commissionEarned',
                'bonusUnlocked', 'milestoneReached', 'referralNotifications',
                'notifyNewReferral', 'notifyCommission', 'notifyMilestone'
            ],

            'network' => [
                'networkDashboard', 'networkStatistics', 'networkMainContent', 'maxLevels',
                'networkHierarchy', 'clickToExpand', 'allNodesExpanded', 'allNodesCollapsed',
                'noReferralsYet', 'viewReferralCode', 'referralBonuses', 'achieveTheseBonus',
                'milestoneBonuses', 'achievementBonuses', 'loyaltyBonuses', 'claimed', 'eligible',
                'reward', 'recurring', 'every', 'requires', 'expandOrCollapse', 'howItWorks',
                'shareReferralCode', 'whenTheySignUp', 'earnCommissions', 'buildDeeperNetwork',
                'networkLevels', 'level1Direct', 'level2', 'level3', 'level4Plus', 'referralLevel',
                'referralLevelDashboard', 'currentLevel', 'yourActiveCommissionRate', 'yourReferrals',
                'levelProgress', 'previousLevel', 'completedMilestone', 'requiredReferrals',
                'requiredInvestment', 'nextLevel', 'unlockHigherCommission', 'inProgress',
                'targetCommission', 'referralsProgress', 'moreReferralsNeeded', 'moreInvestmentNeeded',
                'requirementsToUnlock', 'refer', 'activeMembersAnd', 'achieve', 'inTeamInvestment',
                'maxLevel', 'maximumLevelReached', 'highestCommissionRate', 'viewLevel'
            ],

            'header' => [
                'home', 'about', 'aboutUs', 'features', 'howItWorks', 'plans', 'pricingPlans',
                'investmentPlans', 'faq', 'faqs', 'contact', 'contactUs', 'blog', 'news',
                'support', 'helpCenter', 'login', 'signIn', 'register', 'signUp', 'getStarted',
                'dashboard', 'myAccount', 'profile', 'settings', 'logout', 'signOut',
                'language', 'selectLanguage', 'theme', 'darkMode', 'lightMode', 'notifications',
                'messages', 'wallet', 'balance', 'deposit', 'withdraw', 'invest', 'trade',
                'portfolio', 'history', 'transactions', 'referrals', 'rewards', 'menu',
                'mobileMenu', 'closeMenu', 'search', 'searchPlaceholder', 'viewProfile',
                'accountSettings', 'billingSettings', 'securitySettings', 'privacySettings',
                'helpSupport', 'documentation', 'termsOfService', 'privacyPolicy',
                'cookiePolicy', 'welcomeBack', 'headerWalletConnect', 'headerConnectWallet',
                'headerDisconnectWallet', 'headerWalletConnected', 'headerSwitchAccount',
                'headerAccountDropdown', 'headerSelectAccount', 'headerNoWalletFound',
                'headerInstallWallet', 'headerWalletOptions', 'headerConnecting',
                'headerSelectWallet', 'headerChooseWallet', 'headerMetaMask', 'headerTrustWallet',
                'headerWalletConnect', 'headerPopularWallets', 'headerRecommended',
                'headerCompatibleWallets', 'headerBackToLogin', 'headerRegisterWithWallet',
                'headerLoginWithWallet', 'headerWeb3Login', 'headerWeb3Register',
                'walletModalTitle', 'walletModalSubtitle', 'walletModalAlreadyHaveAccount',
                'walletModalLoginHere', 'walletModalNewUser', 'walletModalRegisterHere',
                'walletModalLoginWithEmail', 'walletModalRegisterWithEmail',
                'walletModalEnterEmail', 'walletModalEnterPassword', 'walletModalPassword',
                'walletModalEmailPlaceholder', 'walletModalPasswordPlaceholder',
                'walletModalForgotPassword', 'walletModalRememberMe', 'walletModalLoginButton',
                'walletModalRegisterButton', 'walletModalOrContinueWith', 'walletModalConnectWallet',
                'walletModalChooseWalletProvider', 'walletModalLoggingIn', 'walletModalConnecting',
                'walletModalPleaseWait', 'walletModalMetaMask', 'walletModalTrustWallet',
                'walletModalWalletConnect', 'walletModalCoinbase', 'walletModalApproveConnection',
                'walletModalNoAccounts', 'walletModalAuthenticatingWallet',
                'walletModalConnectionRejected', 'walletModalConnectionTimeout',
                'walletModalConnectionError', 'walletModalGettingNonce', 'walletModalSigningMessage',
                'walletModalConfirmInWallet', 'walletModalNonceFailed', 'walletModalSignatureFailed',
                'walletModalSignatureRejected', 'walletModalPleaseWait', 'walletModalLoginTab',
                'walletModalRegisterTab', 'walletModalNameLabel', 'walletModalNamePlaceholder',
                'walletModalConfirmPasswordLabel', 'walletModalConfirmPasswordPlaceholder',
                'walletModalCreatingAccount', 'walletModalFillAllFields', 'walletModalPasswordMismatch',
                'walletModalPasswordTooShort', 'walletModalRegistrationFailed',
                'walletModalBackToWallets', 'walletModalCheckEmail'
            ],

            'footer' => [
                'footerAllRightsReserved', 'footerPrivacyPolicy', 'footerTermsService',
                'footerCookiePolicy', 'platformNumber', 'about', 'aboutUs', 'company',
                'ourStory', 'team', 'careers', 'press', 'blog', 'products', 'features',
                'solutions', 'pricing', 'security', 'resources', 'documentation', 'helpCenter',
                'communityForum', 'statusPage', 'legal', 'termsOfService', 'privacyPolicy',
                'cookiePolicy', 'compliance', 'licenses', 'social', 'followUs', 'facebook',
                'twitter', 'linkedin', 'instagram', 'youtube', 'telegram', 'discord',
                'contact', 'contactUs', 'support', 'salesInquiry', 'partnerships',
                'newsletter', 'subscribeNewsletter', 'enterEmail', 'subscribe', 'stayUpdated',
                'latestNews', 'copyright', 'allRightsReserved', 'madeWith', 'poweredBy'
            ],

            'support' => [
                'filesSelected', 'fileRemoved', 'messageRequired', 'replySubmitted',
                'downloadStarted', 'downloadFailed', 'categoryAccount', 'categoryInvestment',
                'categoryWithdrawal', 'categoryDeposit', 'categoryStaking', 'categoryReferral',
                'categoryTechnical', 'categoryBilling', 'categoryKyc', 'categorySecurity',
                'categoryGeneral', 'categoryFeature', 'priorityLowDescription',
                'priorityMediumDescription', 'priorityHighDescription', 'priorityUrgentDescription',
                'filesAttached', 'subjectRequired', 'descriptionRequired', 'categoryRequired',
                'ticketCreatedSuccess', 'ticketCreatedError', 'attachFiles', 'maxFileSize',
                'attachedFiles', 'enterSubject', 'enterDescription', 'kyc', 'general', 'feature',
                'createSupportTicket', 'describeProblemGetHelp', 'back', 'enterTicketSubject',
                'category', 'selectCategory', 'technical', 'billing', 'account', 'other',
                'priority', 'low', 'medium', 'high', 'urgent', 'description', 'describeYourProblem',
                'attachments', 'maxFileSize10MB', 'supportedFormats', 'selectedFiles', 'creating',
                'createTicket', 'helpfulTips', 'beSpecific', 'provideDetailedDescription',
                'attachScreenshots', 'visualsHelpUsUnderstand', 'setPriority',
                'urgentForCriticalIssues', 'responseTime', 'weRespondWithin24Hours',
                'pleaseFillRequiredFields', 'ticketCreatedSuccessfully', 'failedToCreateTicket',
                'supportTickets', 'manageYourSupportRequests', 'searchTickets', 'allStatus',
                'open', 'inProgress', 'resolved', 'closed', 'clearFilters', 'ticket', 'reply',
                'replies', 'noTicketsFound', 'createYourFirstTicket', 'showing', 'of', 'results',
                'previous', 'next', 'support', 'helpCenter'
            ],

            'kyc_verification' => [
                'verifyIdentityToUnlock', 'verificationPending', 'underReview',
                'verificationApproved', 'verificationRejected', 'unknownStatus',
                'pendingDescription', 'reviewingDescription', 'approvedDescription',
                'rejectedDescription', 'statusUnavailable', 'rejectionReason', 'submitted',
                'reviewed', 'resubmitKycDocuments', 'submitKycDocuments',
                'provideAccurateInformation', 'personalInformation', 'firstName',
                'enterFirstName', 'lastName', 'enterLastName', 'dateOfBirth', 'phoneNumber',
                'enterPhoneNumber', 'addressInformation', 'address', 'enterAddress', 'city',
                'enterCity', 'stateProvince', 'enterState', 'postalCode', 'enterPostalCode',
                'country', 'selectCountry', 'documentInformation', 'documentType',
                'selectDocumentType', 'passport', 'driversLicense', 'nationalId',
                'documentNumber', 'enterDocumentNumber', 'documentUpload', 'documentFrontSide',
                'uploadDocumentFrontSide', 'clickToUploadFrontSide', 'pngJpgUpTo8MB',
                'documentFrontPreview', 'removeDocumentFrontImage', 'documentBackSide',
                'uploadDocumentBackSide', 'clickToUploadBackSide', 'documentBackPreview',
                'removeDocumentBackImage', 'selfieWithDocument', 'uploadSelfieWithDocument',
                'clickToUploadSelfie', 'holdDocumentNextToFace', 'selfiePreview',
                'removeSelfieImage', 'agreeToTermsText', 'agreeToTermsRequired', 'submittingKyc',
                'resubmitKyc', 'submitKyc', 'submitting', 'selectImageFileOnly',
                'onlyJpegPngAllowed', 'fileUploadedSuccessfully', 'imageCompressedSuccessfully',
                'failedToCompressImage', 'failedToLoadImage', 'fileRemovedSuccessfully',
                'fillAllRequiredFields', 'kycSubmittedSuccessfully', 'failedToSubmitKyc',
                'unexpectedError', 'kycVerification', 'verifyYourIdentity', 'kycApproved',
                'kycUnderReview', 'gender', 'male', 'female', 'selectGender'
            ],

            'login_history' => [
                'loginHistory', 'loginStatistics', 'totalLoginAttempts', 'totalAttempts',
                'successfulLogins', 'failedLoginAttempts', 'failedAttempts', 'uniqueIpAddresses',
                'recentLoginActivity', 'refreshingLoginHistory', 'refreshLoginHistory',
                'loginHistoryTable', 'dateTime', 'ipAddress', 'location', 'device', 'browser',
                'deviceType', 'loginSuccessfulAt', 'loginFailedAt', 'currentSession',
                'unknownOs', 'noLoginHistoryFound', 'loginAttemptsWillAppearHere',
                'loginAttemptDetails', 'platform', 'failureReason', 'userAgent', 'blocking',
                'blockIp', 'securityAlert', 'recentFailedAttemptsDetected', 'enable2FA',
                'loginHistoryRefreshed', 'failedToRefreshLoginHistory', 'confirmBlockIp',
                'ipAddressBlocked', 'failedToBlockIp'
            ],

            'password_change' => [
                'updateAccountPassword', 'passwordRequirements', 'atLeast8Characters',
                'containsUppercaseLowercase', 'containsAtLeastOneNumber',
                'containsSpecialCharacter', 'currentPassword', 'enterCurrentPassword',
                'hideCurrentPassword', 'showCurrentPassword', 'newPassword', 'enterNewPassword',
                'hideNewPassword', 'showNewPassword', 'confirmNewPassword', 'confirmNewPassword2',
                'hidePasswordConfirmation', 'showPasswordConfirmation', 'passwordStrengthVeryWeak',
                'passwordStrengthWeak', 'passwordStrengthFair', 'passwordStrengthGood',
                'passwordStrengthStrong', 'passwordStrength', 'atLeast8CharactersShort',
                'upperLowercaseLetters', 'atLeastOneNumber', 'specialCharacter', 'passwordsMatch',
                'passwordsDoNotMatch', 'pleaseFixFollowingErrors', 'resetForm', 'resetFormClear',
                'updatePassword', 'updatingPassword', 'updating', 'passwordMustBe8Characters',
                'passwordMustContainMixedCase', 'passwordMustContainNumber',
                'passwordMustContainSpecialChar', 'passwordConfirmationNoMatch',
                'ensureAllFieldsFilled', 'enterCurrentPasswordPlease', 'currentPasswordTooShort',
                'newPasswordMustBeDifferent', 'passwordUpdatedSuccessfully',
                'currentPasswordIncorrect', 'failedToUpdatePassword', 'passwordTooCommon',
                'avoidSequentialCharacters', 'avoidRepeatedCharacters', 'securityTips',
                'useUniquePassword', 'dontReusePasswords', 'enableTwoFA', 'addExtraSecurity',
                'regularUpdates', 'changePasswordRegularly', 'avoidSharing', 'neverSharePassword'
            ],

            'two_factor_auth' => [
                'twoFactorAuthentication', 'addExtraSecurityLayer', 'enabled', 'disabled',
                'manage2FA', 'setup2FA', 'accountIsProtected', 'secureYourAccount',
                'setupInstructions', 'installAuthenticatorApp', 'scanQRCodeOrEnterSecret',
                'enter6DigitCodeFromApp', 'saveRecoveryCodesSafely', 'qrCodeAlt',
                'scanQRCodeWithApp', 'copySecretKeyInstead', 'orEnterCodeManually',
                'copySecretToClipboard', 'enterVerificationCodeFromApp',
                'enter6DigitVerificationCode', 'enabling', 'enableTwoFactorAuth',
                'twoFactorIsActive', 'accountProtectedWith2FA', 'testYour2FA',
                'enterVerificationCodeToTest', 'enter6DigitCodeToTest', 'enter6DigitCode',
                'disableTwoFactorAuth', 'removeExtraSecurityLayer', 'enterPasswordToDisable',
                'enterPasswordToConfirmDisabling', 'enterPasswordToConfirm', 'hidePassword',
                'showPassword', 'disabling', 'disable2FA', 'recoveryCodes',
                'useIfLoseAccess', 'important', 'saveCodesSecurely',
                'downloadRecoveryCodesAsFile', 'downloadRecoveryCodes',
                'generateNewRecoveryCodes', 'generating', 'generateNewCodes',
                'enable2FAToSeeRecoveryCodes', 'useAuthenticatorApp',
                'downloadGoogleAuthOrAuthy', 'backupRecoveryCodes', 'storeCodesSecureLocation',
                'testRegularly', 'verifyCodesWork', 'keepDeviceSecure', 'protectPhoneWithLock',
                'noSecretAvailable', 'secretKeyCopied', 'failedToCopySecret',
                'enterValid6DigitCode', 'twoFactorEnabledSuccessfully', 'failedToEnable2FA',
                'codeVerifiedSuccessfully', 'invalidVerificationCode',
                'twoFactorDisabledSuccessfully', 'failedToDisable2FA',
                'noRecoveryCodesAvailable', 'twoFactorRecoveryCodes', 'generated',
                'recoveryCodesImportant', 'storeCodesSecurely', 'recoveryCodesDownloaded',
                'failedToDownloadCodes', 'enable2FAFirst', 'newRecoveryCodesGenerated',
                'failedToGenerateNewCodes'
            ],

            'ticket_details' => [
                'createdOn', 'unknownUser', 'conversation', 'noRepliesYet', 'addReply',
                'typeYourReply', 'sendReply', 'ticketClosedNoReply', 'ticketDetails',
                'ticketNumber', 'replyToTicket', 'viewAllTickets', 'createNewTicket',
                'supportInfo', 'within24Hours', 'emergencySupport', 'forUrgentIssues',
                'helpCenter', 'selfServiceOptions', 'ticketActivity', 'totalReplies',
                'lastActivity', 'day', 'hour', 'hours', 'minute', 'minutes', 'ago',
                'pleaseEnterReplyMessage', 'replySentSuccessfully', 'failedToSendReply',
                'quickActions'
            ],

            'subscription' => [
                'paid', 'subscriptionPlans', 'subscription', 'currentSubscription', 'planName',
                'monthlyPrice', 'month', 'expiresOn', 'daysLeft', 'hoursLeft', 'planBenefits',
                'noMiningFees', 'reducedMiningFees', 'miningRateMultiplier',
                'cancelSubscription', 'subscribe', 'freeUser', 'freeUserDescription', 'features',
                'basic', 'availablePlans', 'choosePlanThatSuitsYou', 'perMonth',
                'noPlansAvailable', 'checkBackLaterForPlans', 'confirmSubscription',
                'chargedMonthly', 'paymentMethod', 'walletBalance', 'areYouSure',
                'cancelSubscriptionWarning', 'keepSubscription', 'confirmCancel',
                'subscriptionSuccessful', 'subscriptionFailed', 'subscriptionCancelled',
                'cancellationFailed', 'subscriptionStatus', 'manageSubscription', 'upgradePlan',
                'currentPlan', 'miningFee'
            ],

            'miscellaneous' => [
                'free', 'xp', 'startMiningToUnlock', 'entryFee', 'grossAmount',
                'subscriptionDiscountApplied', 'upgradeToReduceFees', 'joined_date',
                'resetPasswordFor', 'resettingPasswordFor', 'paymentInstructions',
                'withdrawalInstructions', 'paymentCredentials', 'optional', 'acceptTerms',
                'acceptedOn', 'acceptTermsAndConditions', 'accountVerification',
                'activeUsers', 'additionalInfo', 'agreeToTerms', 'allData', 'allowNotifications',
                'apply', 'assetTokenization', 'available', 'availableAssets', 'bio',
                'blockchainTechnology', 'browsingData', 'businessModel', 'businessModelText',
                'chargesAndFees', 'chargesAndFeesText', 'chooseYourCurrency', 'collectInfo',
                'complianceNotice', 'complianceNoticeText', 'complyCrypto', 'complyKyc',
                'complyMlft', 'complyTaxRegulation', 'connectWallet', 'connectWalletDescription',
                'connectYourWallet', 'connecting', 'cookieConsent', 'cookiePreferences',
                'cookieSettings', 'cookies', 'descriptionOfService', 'descriptionOfServiceText',
                'deviceInfo', 'documentBackImage', 'documentFrontImage', 'documentNumber',
                'documentType', 'driversLicense', 'emailAddress', 'essentialOnly',
                'failedToConnectWallet', 'feesAndPayments', 'feesAndPaymentsText', 'female',
                'financialRisks', 'financialRisksText', 'gender', 'howWeUseInfo',
                'howWeUseInfoText', 'installMetaMask', 'installTrustWallet', 'jpgPngMax2mb',
                'kycApproved', 'kycUnderReview', 'kycVerification', 'limitationOfLiability',
                'limitationOfLiabilityText', 'liquidStaking', 'liveOpportunities', 'loggingIn',
                'loginFailed', 'maintainSecurity', 'male', 'marketVolatility', 'memberSince',
                'mobileFirstWeb3Wallet', 'nationalId', 'noAccountsFound', 'noFraudulent',
                'noUnauthorizedAccess', 'noWalletDetected', 'passport', 'paymentDetails',
                'paymentInformation', 'platformStats', 'pleaseApproveInWallet',
                'popularEthereumWallet', 'profileInformation', 'profilePhoto',
                'provideAccurateInfo', 'realEstate', 'redirecting', 'regulatoryChanges',
                'secureConnection', 'secureConnectionDesc', 'selectDocumentType', 'selectGender',
                'selectOption', 'selfieWithDocument', 'settings', 'smartContractVulnerabilities',
                'submitKyc', 'submitting', 'technologyFailures', 'termination', 'terminationText',
                'terms', 'totalValueLocked', 'tryAgain', 'updateProfile', 'updating', 'usageData',
                'userId', 'userResponsibilities', 'userResponsibilitiesText', 'verifyYourIdentity',
                'walletAddress', 'walletAddresses', 'walletConfigNotFound', 'walletConnected',
                'walletConnection', 'walletConnectionText', 'walletProviderNotFound',
                'weValueYourPrivacy', 'willBeRefunded', 'withdrawalInformation', 'yieldFarming',
                'totalWithdraw'
            ],
        ];

        return $components[$component] ?? [];
    }
}
